<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.


/**
 * Core external functions and service definitions.
 *
 * The functions and services defined on this file are
 * processed and registered into the Moodle DB after any
 * install or upgrade operation. All plugins support this.
 *
 * For more information, take a look to the documentation available:
 *     - Webservices API: {@link http://docs.moodle.org/dev/Web_services_API}
 *     - External API: {@link http://docs.moodle.org/dev/External_functions_API}
 *     - Upgrade API: {@link http://docs.moodle.org/dev/Upgrade_API}
 *
 * @package    core_webservice
 * @category   webservice
 * @copyright  2009 Petr Skodak
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

$functions = array(
    'core_auth_confirm_user' => array(
        'classname'   => 'core_auth_external',
        'methodname'  => 'confirm_user',
        'description' => 'Confirm a user account.',
        'type'        => 'write',
        'ajax'          => true,
        'loginrequired' => false,
    ),
    'core_auth_request_password_reset' => array(
        'classname'   => 'core_auth_external',
        'methodname'  => 'request_password_reset',
        'description' => 'Requests a password reset.',
        'type'        => 'write',
        'ajax'          => true,
        'loginrequired' => false,
    ),
    'core_auth_is_minor' => array(
        'classname'   => 'core_auth_external',
        'methodname'  => 'is_minor',
        'description' => 'Requests a check if a user is a digital minor.',
        'type'        => 'read',
        'ajax'          => true,
        'loginrequired' => false,
    ),
    'core_auth_is_age_digital_consent_verification_enabled' => array(
        'classname'   => 'core_auth_external',
        'methodname'  => 'is_age_digital_consent_verification_enabled',
        'description' => 'Checks if age digital consent verification is enabled.',
        'type'        => 'read',
        'ajax'          => true,
        'loginrequired' => false,
    ),
    'core_auth_resend_confirmation_email' => array(
        'classname'   => 'core_auth_external',
        'methodname'  => 'resend_confirmation_email',
        'description' => 'Resend confirmation email.',
        'type'        => 'write',
        'ajax'          => true,
        'loginrequired' => false,
    ),
    'core_backup_get_async_backup_progress' => array(
        'classname'   => 'core_backup_external',
        'classpath' => 'backup/externallib.php',
        'methodname'  => 'get_async_backup_progress',
        'description' => 'Get the progress of an Asyncronhous backup.',
        'type'        => 'read',
        'ajax'          => true,
        'loginrequired' => true,
    ),
    'core_backup_get_async_backup_links_backup' => array(
        'classname'   => 'core_backup_external',
        'classpath' => 'backup/externallib.php',
        'methodname'  => 'get_async_backup_links_backup',
        'description' => 'Gets the data to use when updating the status table row in the UI for when an async backup completes.',
        'type'        => 'read',
        'ajax'          => true,
        'loginrequired' => true,
   ),
   'core_backup_get_async_backup_links_restore' => array(
        'classname'   => 'core_backup_external',
        'classpath' => 'backup/externallib.php',
        'methodname'  => 'get_async_backup_links_restore',
        'description' => 'Gets the data to use when updating the status table row in the UI for when an async restore completes.',
        'type'        => 'read',
        'ajax'          => true,
        'loginrequired' => true,
    ),
    'core_backup_get_copy_progress' => array(
        'classname'   => 'core_backup_external',
        'classpath' => 'backup/externallib.php',
        'methodname'  => 'get_copy_progress',
        'description' => 'Gets the progress of course copy operations.',
        'type'        => 'read',
        'ajax'          => true,
        'loginrequired' => true,
    ),
    'core_backup_submit_copy_form' => array(
        'classname'   => 'core_backup_external',
        'classpath' => 'backup/externallib.php',
        'methodname'  => 'submit_copy_form',
        'description' => 'Handles ajax submission of course copy form.',
        'type'        => 'read',
        'ajax'          => true,
        'loginrequired' => true,
    ),
    'core_badges_get_user_badges' => array(
        'classname'     => 'core_badges_external',
        'methodname'    => 'get_user_badges',
        'description'   => 'Returns the list of badges awarded to a user.',
        'type'          => 'read',
        'capabilities'  => 'moodle/badges:viewotherbadges',
        'services'      => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_blog_get_entries' => array(
        'classname'   => 'core_blog\external',
        'methodname'  => 'get_entries',
        'description' => 'Returns blog entries.',
        'type'        => 'read',
        'services'    => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax'          => true,
        'loginrequired' => false,
    ),
    'core_blog_view_entries' => array(
        'classname'   => 'core_blog\external',
        'methodname'  => 'view_entries',
        'description' => 'Trigger the blog_entries_viewed event.',
        'type'        => 'read',
        'services'    => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax'          => true,
        'loginrequired' => false,
    ),
    'core_calendar_get_calendar_monthly_view' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'get_calendar_monthly_view',
        'description' => 'Fetch the monthly view data for a calendar',
        'classpath' => 'calendar/externallib.php',
        'type' => 'read',
        'capabilities' => '',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_calendar_get_calendar_day_view' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'get_calendar_day_view',
        'description' => 'Fetch the day view data for a calendar',
        'classpath' => 'calendar/externallib.php',
        'type' => 'read',
        'capabilities' => '',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_calendar_get_calendar_upcoming_view' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'get_calendar_upcoming_view',
        'description' => 'Fetch the upcoming view data for a calendar',
        'classpath' => 'calendar/externallib.php',
        'type' => 'read',
        'capabilities' => '',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_calendar_update_event_start_day' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'update_event_start_day',
        'description' => 'Update the start day (but not time) for an event.',
        'classpath' => 'calendar/externallib.php',
        'type' => 'write',
        'capabilities' => 'moodle/calendar:manageentries, moodle/calendar:manageownentries, moodle/calendar:managegroupentries',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_calendar_create_calendar_events' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'create_calendar_events',
        'description' => 'Create calendar events',
        'classpath' => 'calendar/externallib.php',
        'type' => 'write',
        'capabilities' => 'moodle/calendar:manageentries, moodle/calendar:manageownentries, moodle/calendar:managegroupentries',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_calendar_delete_calendar_events' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'delete_calendar_events',
        'description' => 'Delete calendar events',
        'classpath' => 'calendar/externallib.php',
        'type' => 'write',
        'capabilities' => 'moodle/calendar:manageentries, moodle/calendar:manageownentries, moodle/calendar:managegroupentries',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_calendar_get_calendar_events' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'get_calendar_events',
        'description' => 'Get calendar events',
        'classpath' => 'calendar/externallib.php',
        'type' => 'read',
        'capabilities' => 'moodle/calendar:manageentries, moodle/calendar:manageownentries, moodle/calendar:managegroupentries',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_calendar_get_action_events_by_timesort' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'get_calendar_action_events_by_timesort',
        'description' => 'Get calendar action events by tiemsort',
        'classpath' => 'calendar/externallib.php',
        'type' => 'read',
        'capabilities' => 'moodle/calendar:manageentries, moodle/calendar:manageownentries, moodle/calendar:managegroupentries',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_calendar_get_action_events_by_course' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'get_calendar_action_events_by_course',
        'description' => 'Get calendar action events by course',
        'classpath' => 'calendar/externallib.php',
        'type' => 'read',
        'capabilities' => 'moodle/calendar:manageentries, moodle/calendar:manageownentries, moodle/calendar:managegroupentries',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_calendar_get_action_events_by_courses' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'get_calendar_action_events_by_courses',
        'description' => 'Get calendar action events by courses',
        'classpath' => 'calendar/externallib.php',
        'type' => 'read',
        'capabilities' => 'moodle/calendar:manageentries, moodle/calendar:manageownentries, moodle/calendar:managegroupentries',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_calendar_get_calendar_event_by_id' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'get_calendar_event_by_id',
        'description' => 'Get calendar event by id',
        'classpath' => 'calendar/externallib.php',
        'type' => 'read',
        'capabilities' => 'moodle/calendar:manageentries, moodle/calendar:manageownentries, moodle/calendar:managegroupentries',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_calendar_submit_create_update_form' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'submit_create_update_form',
        'description' => 'Submit form data for event form',
        'classpath' => 'calendar/externallib.php',
        'type' => 'write',
        'capabilities' => 'moodle/calendar:manageentries, moodle/calendar:manageownentries, moodle/calendar:managegroupentries',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_calendar_get_calendar_access_information' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'get_calendar_access_information',
        'description' => 'Convenience function to retrieve some permissions/access information for the given course calendar.',
        'classpath' => 'calendar/externallib.php',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_calendar_get_allowed_event_types' => array(
        'classname' => 'core_calendar_external',
        'methodname' => 'get_allowed_event_types',
        'description' => 'Get the type of events a user can create in the given course.',
        'classpath' => 'calendar/externallib.php',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_calendar_get_timestamps' => [
        'classname'     => 'core_calendar_external',
        'methodname'    => 'get_timestamps',
        'description'   => 'Fetch unix timestamps for given date times.',
        'classpath' => 'calendar/externallib.php',
        'type'          => 'read',
        'ajax'          => true,
    ],
    'core_calendar_get_calendar_export_token' => [
        'classname'     => 'core_calendar\external\export\token',
        'description'   => 'Return the auth token required for exporting a calendar.',
        'type'          => 'read',
        'services'      => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ],
    'core_cohort_add_cohort_members' => array(
        'classname' => 'core_cohort_external',
        'methodname' => 'add_cohort_members',
        'classpath' => 'cohort/externallib.php',
        'description' => 'Adds cohort members.',
        'type' => 'write',
        'capabilities' => 'moodle/cohort:assign'
    ),
    'core_cohort_create_cohorts' => array(
        'classname' => 'core_cohort_external',
        'methodname' => 'create_cohorts',
        'classpath' => 'cohort/externallib.php',
        'description' => 'Creates new cohorts.',
        'type' => 'write',
        'capabilities' => 'moodle/cohort:manage'
    ),
    'core_cohort_delete_cohort_members' => array(
        'classname' => 'core_cohort_external',
        'methodname' => 'delete_cohort_members',
        'classpath' => 'cohort/externallib.php',
        'description' => 'Deletes cohort members.',
        'type' => 'write',
        'capabilities' => 'moodle/cohort:assign'
    ),
    'core_cohort_delete_cohorts' => array(
        'classname' => 'core_cohort_external',
        'methodname' => 'delete_cohorts',
        'classpath' => 'cohort/externallib.php',
        'description' => 'Deletes all specified cohorts.',
        'type' => 'write',
        'capabilities' => 'moodle/cohort:manage'
    ),
    'core_cohort_get_cohort_members' => array(
        'classname' => 'core_cohort_external',
        'methodname' => 'get_cohort_members',
        'classpath' => 'cohort/externallib.php',
        'description' => 'Returns cohort members.',
        'type' => 'read',
        'capabilities' => 'moodle/cohort:view'
    ),
    'core_cohort_search_cohorts' => array(
        'classname' => 'core_cohort_external',
        'methodname' => 'search_cohorts',
        'classpath' => 'cohort/externallib.php',
        'description' => 'Search for cohorts.',
        'type' => 'read',
        'ajax' => true,
        'capabilities' => 'moodle/cohort:view'
    ),
    'core_cohort_get_cohorts' => array(
        'classname' => 'core_cohort_external',
        'methodname' => 'get_cohorts',
        'classpath' => 'cohort/externallib.php',
        'description' => 'Returns cohort details.',
        'type' => 'read',
        'capabilities' => 'moodle/cohort:view'
    ),
    'core_cohort_update_cohorts' => array(
        'classname' => 'core_cohort_external',
        'methodname' => 'update_cohorts',
        'classpath' => 'cohort/externallib.php',
        'description' => 'Updates existing cohorts.',
        'type' => 'write',
        'capabilities' => 'moodle/cohort:manage'
    ),
    'core_comment_get_comments' => array(
        'classname' => 'core_comment_external',
        'methodname' => 'get_comments',
        'description' => 'Returns comments.',
        'type' => 'read',
        'capabilities' => 'moodle/comment:view',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_comment_add_comments' => array(
        'classname' => 'core_comment_external',
        'methodname' => 'add_comments',
        'description' => 'Adds a comment or comments.',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_comment_delete_comments' => array(
        'classname' => 'core_comment_external',
        'methodname' => 'delete_comments',
        'description' => 'Deletes a comment or comments.',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_completion_get_activities_completion_status' => array(
        'classname' => 'core_completion_external',
        'methodname' => 'get_activities_completion_status',
        'description' => 'Return the activities completion status for a user in a course.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_completion_get_course_completion_status' => array(
        'classname' => 'core_completion_external',
        'methodname' => 'get_course_completion_status',
        'description' => 'Returns course completion status.',
        'type' => 'read',
        'capabilities' => 'report/completion:view',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_completion_mark_course_self_completed' => array(
        'classname' => 'core_completion_external',
        'methodname' => 'mark_course_self_completed',
        'description' => 'Update the course completion status for the current user (if course self-completion is enabled).',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_completion_update_activity_completion_status_manually' => array(
        'classname' => 'core_completion_external',
        'methodname' => 'update_activity_completion_status_manually',
        'description' => 'Update completion status for the current user in an activity, only for activities with manual tracking.',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true,
    ),
    'core_completion_override_activity_completion_status' => array(
        'classname'     => 'core_completion_external',
        'methodname'    => 'override_activity_completion_status',
        'description'   => 'Update completion status for a user in an activity by overriding it.',
        'type'          => 'write',
        'capabilities'  => 'moodle/course:overridecompletion',
        'ajax'          => true,
    ),
    'core_course_create_categories' => array(
        'classname' => 'core_course_external',
        'methodname' => 'create_categories',
        'classpath' => 'course/externallib.php',
        'description' => 'Create course categories',
        'type' => 'write',
        'capabilities' => 'moodle/category:manage'
    ),
    'core_course_create_courses' => array(
        'classname' => 'core_course_external',
        'methodname' => 'create_courses',
        'classpath' => 'course/externallib.php',
        'description' => 'Create new courses',
        'type' => 'write',
        'capabilities' => 'moodle/course:create, moodle/course:visibility'
    ),
    'core_course_delete_categories' => array(
        'classname' => 'core_course_external',
        'methodname' => 'delete_categories',
        'classpath' => 'course/externallib.php',
        'description' => 'Delete course categories',
        'type' => 'write',
        'capabilities' => 'moodle/category:manage'
    ),
    'core_course_delete_courses' => array(
        'classname' => 'core_course_external',
        'methodname' => 'delete_courses',
        'classpath' => 'course/externallib.php',
        'description' => 'Deletes all specified courses',
        'type' => 'write',
        'capabilities' => 'moodle/course:delete'
    ),
    'core_course_delete_modules' => array(
        'classname' => 'core_course_external',
        'methodname' => 'delete_modules',
        'classpath' => 'course/externallib.php',
        'description' => 'Deletes all specified module instances',
        'type' => 'write',
        'capabilities' => 'moodle/course:manageactivities'
    ),
    'core_course_duplicate_course' => array(
        'classname' => 'core_course_external',
        'methodname' => 'duplicate_course',
        'classpath' => 'course/externallib.php',
        'description' => 'Duplicate an existing course (creating a new one).',
        'type' => 'write',
        'capabilities' => 'moodle/backup:backupcourse, moodle/restore:restorecourse, moodle/course:create'
    ),
    'core_course_get_categories' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_categories',
        'classpath' => 'course/externallib.php',
        'description' => 'Return category details',
        'type' => 'read',
        'capabilities' => 'moodle/category:viewhiddencategories',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_contents' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_course_contents',
        'classpath' => 'course/externallib.php',
        'description' => 'Get course contents',
        'type' => 'read',
        'capabilities' => 'moodle/course:update, moodle/course:viewhiddencourses',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_course_module' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_course_module',
        'classpath' => 'course/externallib.php',
        'description' => 'Return information about a course module',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_course_module_by_instance' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_course_module_by_instance',
        'classpath' => 'course/externallib.php',
        'description' => 'Return information about a given module name and instance id',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_module' => array(
        'classname'   => 'core_course_external',
        'methodname'  => 'get_module',
        'classpath'   => 'course/externallib.php',
        'description' => 'Returns html with one activity module on course page',
        'type'        => 'read',
        'ajax'        => true,
    ),
    'core_course_edit_module' => array(
        'classname'   => 'core_course_external',
        'methodname'  => 'edit_module',
        'classpath'   => 'course/externallib.php',
        'description' => 'Performs an action on course module (change visibility, duplicate, delete, etc.)',
        'type'        => 'write',
        'ajax'        => true,
    ),
    'core_course_edit_section' => array(
        'classname'   => 'core_course_external',
        'methodname'  => 'edit_section',
        'classpath'   => 'course/externallib.php',
        'description' => 'Performs an action on course section (change visibility, set marker, delete)',
        'type'        => 'write',
        'ajax'        => true,
    ),
    'core_course_get_courses' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_courses',
        'classpath' => 'course/externallib.php',
        'description' => 'Return course details',
        'type' => 'read',
        'capabilities' => 'moodle/course:view, moodle/course:update, moodle/course:viewhiddencourses',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_import_course' => array(
        'classname' => 'core_course_external',
        'methodname' => 'import_course',
        'classpath' => 'course/externallib.php',
        'description' => 'Import course data from a course into another course. Does not include any user data.',
        'type' => 'write',
        'capabilities' => 'moodle/backup:backuptargetimport, moodle/restore:restoretargetimport'
    ),
    'core_course_search_courses' => array(
        'classname' => 'core_course_external',
        'methodname' => 'search_courses',
        'classpath' => 'course/externallib.php',
        'description' => 'Search courses by (name, module, block, tag)',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_update_categories' => array(
        'classname' => 'core_course_external',
        'methodname' => 'update_categories',
        'classpath' => 'course/externallib.php',
        'description' => 'Update categories',
        'type' => 'write',
        'capabilities' => 'moodle/category:manage',
    ),
    'core_course_update_courses' => array(
        'classname' => 'core_course_external',
        'methodname' => 'update_courses',
        'classpath' => 'course/externallib.php',
        'description' => 'Update courses',
        'type' => 'write',
        'capabilities' => 'moodle/course:update, moodle/course:changecategory, moodle/course:changefullname, '
            . 'moodle/course:changeshortname, moodle/course:changeidnumber, moodle/course:changesummary, moodle/course:visibility'
    ),
    'core_course_view_course' => array(
        'classname' => 'core_course_external',
        'methodname' => 'view_course',
        'classpath' => 'course/externallib.php',
        'description' => 'Log that the course was viewed',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_user_navigation_options' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_user_navigation_options',
        'classpath' => 'course/externallib.php',
        'description' => 'Return a list of navigation options in a set of courses that are avaialable or not for the current user.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_user_administration_options' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_user_administration_options',
        'classpath' => 'course/externallib.php',
        'description' => 'Return a list of administration options in a set of courses that are avaialable or not for the current
                            user.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_courses_by_field' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_courses_by_field',
        'classpath' => 'course/externallib.php',
        'description' => 'Get courses matching a specific field (id/s, shortname, idnumber, category)',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_check_updates' => array(
        'classname' => 'core_course_external',
        'methodname' => 'check_updates',
        'classpath' => 'course/externallib.php',
        'description' => 'Check if there is updates affecting the user for the given course and contexts.',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_updates_since' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_updates_since',
        'classpath' => 'course/externallib.php',
        'description' => 'Check if there are updates affecting the user for the given course since the given time stamp.',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_enrolled_courses_by_timeline_classification' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_enrolled_courses_by_timeline_classification',
        'classpath' => 'course/externallib.php',
        'description' => 'List of enrolled courses for the given timeline classification (past, inprogress, or future).',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_recent_courses' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_recent_courses',
        'classpath' => 'course/externallib.php',
        'description' => 'List of courses a user has accessed most recently.',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_set_favourite_courses' => array(
        'classname' => 'core_course_external',
        'methodname' => 'set_favourite_courses',
        'classpath' => 'course/externallib.php',
        'description' => 'Add a list of courses to the list of favourite courses.',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_course_get_enrolled_users_by_cmid' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_enrolled_users_by_cmid',
        'classpath' => 'course/externallib.php',
        'description' => 'List users by course module id & filter by group id.',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_course_add_content_item_to_user_favourites' => array(
        'classname' => 'core_course_external',
        'methodname' => 'add_content_item_to_user_favourites',
        'classpath' => 'course/externallib.php',
        'description' => 'Adds a content item (activity, resource or their subtypes) to the favourites for the user.',
        'type' => 'write',
        'ajax' => true,
    ),
    'core_course_remove_content_item_from_user_favourites' => array(
        'classname' => 'core_course_external',
        'methodname' => 'remove_content_item_from_user_favourites',
        'classpath' => 'course/externallib.php',
        'description' => 'Removes a content item (activity, resource or their subtypes) from the favourites for the user.',
        'type' => 'write',
        'ajax' => true,
    ),
    'core_course_get_course_content_items' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_course_content_items',
        'classpath' => 'course/externallib.php',
        'description' => 'Fetch all the content items (activities, resources and their subtypes) for the activity picker',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_course_get_activity_chooser_footer' => array(
        'classname' => 'core_course_external',
        'methodname' => 'get_activity_chooser_footer',
        'classpath' => 'course/externallib.php',
        'description' => 'Fetch the data for the activity chooser footer.',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_course_toggle_activity_recommendation' => array(
        'classname' => 'core_course_external',
        'methodname' => 'toggle_activity_recommendation',
        'classpath' => 'course/externallib.php',
        'description' => 'Adds or removes an activity as a recommendation in the activity chooser.',
        'type' => 'write',
        'ajax' => true,
    ),
    'core_enrol_get_course_enrolment_methods' => array(
        'classname' => 'core_enrol_external',
        'methodname' => 'get_course_enrolment_methods',
        'classpath' => 'enrol/externallib.php',
        'description' => 'Get the list of course enrolment methods',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_enrol_get_enrolled_users' => array(
        'classname' => 'core_enrol_external',
        'methodname' => 'get_enrolled_users',
        'classpath' => 'enrol/externallib.php',
        'description' => 'Get enrolled users by course id.',
        'type' => 'read',
        'capabilities' => 'moodle/user:viewdetails, moodle/user:viewhiddendetails, moodle/course:useremail, moodle/user:update, '
            . 'moodle/site:accessallgroups',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_enrol_get_enrolled_users_with_capability' => array(
        'classname' => 'core_enrol_external',
        'methodname' => 'get_enrolled_users_with_capability',
        'classpath' => 'enrol/externallib.php',
        'description' => 'For each course and capability specified, return a list of the users that are enrolled in the course
                                  and have that capability',
        'type' => 'read',
    ),
    'core_enrol_get_potential_users' => array(
        'classname' => 'core_enrol_external',
        'methodname' => 'get_potential_users',
        'classpath' => 'enrol/externallib.php',
        'description' => 'Get the list of potential users to enrol',
        'ajax' => true,
        'type' => 'read',
        'capabilities' => 'moodle/course:enrolreview'
    ),
    'core_enrol_search_users' => [
        'classname' => 'core_enrol_external',
        'methodname' => 'search_users',
        'classpath' => 'enrol/externallib.php',
        'description' => 'Search within the list of course participants',
        'ajax' => true,
        'type' => 'read',
        'capabilities' => 'moodle/course:viewparticipants',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ],
    'core_enrol_get_users_courses' => array(
        'classname' => 'core_enrol_external',
        'methodname' => 'get_users_courses',
        'classpath' => 'enrol/externallib.php',
        'description' => 'Get the list of courses where a user is enrolled in',
        'type' => 'read',
        'capabilities' => 'moodle/course:viewparticipants',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_enrol_edit_user_enrolment' => array(
        'classname' => 'core_enrol_external',
        'methodname' => 'edit_user_enrolment',
        'classpath' => 'enrol/externallib.php',
        'description' => '** DEPRECATED ** Please do not call this function any more.
                          External function that updates a given user enrolment',
        'type' => 'write',
        'ajax' => true,
    ),
    'core_enrol_submit_user_enrolment_form' => array(
        'classname' => 'core_enrol_external',
        'methodname' => 'submit_user_enrolment_form',
        'classpath' => 'enrol/externallib.php',
        'description' => 'Submit form data for enrolment form',
        'type' => 'write',
        'ajax' => true,
    ),
    'core_enrol_unenrol_user_enrolment' => array(
        'classname' => 'core_enrol_external',
        'methodname' => 'unenrol_user_enrolment',
        'classpath' => 'enrol/externallib.php',
        'description' => 'External function that unenrols a given user enrolment',
        'type' => 'write',
        'ajax' => true,
    ),
    'core_fetch_notifications' => array(
        'classname' => 'core_external',
        'methodname' => 'fetch_notifications',
        'classpath' => 'lib/external/externallib.php',
        'description' => 'Return a list of notifications for the current session',
        'type' => 'read',
        'loginrequired' => false,
        'ajax' => true,
        'readonlysession' => false, // Fetching removes from stack.
    ),
    'core_session_touch' => array(
        'classname' => 'core\session\external',
        'methodname' => 'touch_session',
        'description' => 'Keep the users session alive',
        'type' => 'read',
        'loginrequired' => true,
        'ajax' => true,
    ),
    'core_session_time_remaining' => array(
        'classname' => 'core\session\external',
        'methodname' => 'time_remaining',
        'description' => 'Count the seconds remaining in this session',
        'type' => 'read',
        'loginrequired' => true,
        'ajax' => true,
    ),
    'core_files_get_files' => array(
        'classname' => 'core_files_external',
        'methodname' => 'get_files',
        'description' => 'browse moodle files',
        'type' => 'read',
        'classpath' => 'files/externallib.php',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_files_upload' => array(
        'classname' => 'core_files_external',
        'methodname' => 'upload',
        'description' => 'upload a file to moodle',
        'type'        => 'write',
        'classpath'   => 'files/externallib.php',
    ),
    'core_files_delete_draft_files' => array(
        'classname' => 'core_files\external\delete\draft',
        'description' => 'Delete the indicated files (or directories) from a user draft file area.',
        'type'        => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_files_get_unused_draft_itemid' => array(
        'classname' => 'core_files\external\get\unused_draft',
        'description' => 'Generate a new draft itemid for the current user.',
        'type'        => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_form_get_filetypes_browser_data' => array(
        'classname' => 'core_form\external',
        'methodname' => 'get_filetypes_browser_data',
        'classpath' => '',
        'description' => 'Provides data for the filetypes element browser.',
        'type' => 'read',
        'loginrequired' => false,
        'ajax' => true,
    ),
    'core_form_dynamic_form' => array(
        'classname' => 'core_form\external\dynamic_form',
        'description' => 'Process submission of a dynamic (modal) form',
        'type' => 'write',
        'ajax' => true,
    ),
    'core_get_component_strings' => array(
        'classname' => 'core_external',
        'methodname' => 'get_component_strings',
        'classpath' => 'lib/external/externallib.php',
        'description' => 'Return all raw strings (with {$a->xxx}), for a specific component ' .
            '- similar to core get_component_strings(), call',
        'type' => 'read',
        'loginrequired' => false,
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_get_fragment' => array(
        'classname' => 'core_external',
        'methodname' => 'get_fragment',
        'classpath' => 'lib/external/externallib.php',
        'description' => 'Return a fragment for inclusion, such as a JavaScript page.',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_get_string' => array(
        'classname' => 'core_external',
        'methodname' => 'get_string',
        'classpath' => 'lib/external/externallib.php',
        'description' => 'Return a translated string - similar to core get_string(), call',
        'type' => 'read',
        'loginrequired' => false,
        'ajax' => true,
    ),
    'core_get_strings' => array(
        'classname' => 'core_external',
        'methodname' => 'get_strings',
        'classpath' => 'lib/external/externallib.php',
        'description' => 'Return some translated strings - like several core get_string(), calls',
        'type' => 'read',
        'loginrequired' => false,
        'ajax' => true,
    ),
    'core_get_user_dates' => array(
        'classname' => 'core_external',
        'methodname' => 'get_user_dates',
        'classpath' => 'lib/external/externallib.php',
        'description' => 'Return formatted timestamps',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_grades_get_grades' => array(
        'classname' => 'core_grades_external',
        'methodname' => 'get_grades',
        'description' => '** DEPRECATED ** Please do not call this function any more.
                                     Returns student course total grade and grades for activities.
                                     This function does not return category or manual items.
                                     This function is suitable for managers or teachers not students.',
        'type' => 'read',
        'capabilities' => 'moodle/grade:view, moodle/grade:viewall, moodle/grade:viewhidden'
    ),
    'core_grades_update_grades' => array(
        'classname' => 'core_grades_external',
        'methodname' => 'update_grades',
        'description' => 'Update a grade item and associated student grades.',
        'type' => 'write',
    ),
    'core_grades_grader_gradingpanel_point_fetch' => [
        'classname' => 'core_grades\\grades\\grader\\gradingpanel\\point\\external\\fetch',
        'description' => 'Fetch the data required to display the grader grading panel for simple grading, ' .
            'creating the grade item if required',
        'type' => 'write',
        'ajax' => true,
        'services' => [MOODLE_OFFICIAL_MOBILE_SERVICE],
    ],
    'core_grades_grader_gradingpanel_point_store' => [
        'classname' => 'core_grades\\grades\\grader\\gradingpanel\\point\\external\\store',
        'description' => 'Store the data required to display the grader grading panel for simple grading',
        'type' => 'write',
        'ajax' => true,
        'services' => [MOODLE_OFFICIAL_MOBILE_SERVICE],
    ],
    'core_grades_grader_gradingpanel_scale_fetch' => [
        'classname' => 'core_grades\\grades\\grader\\gradingpanel\\scale\\external\\fetch',
        'description' => 'Fetch the data required to display the grader grading panel for scale-based grading, ' .
            'creating the grade item if required',
        'type' => 'write',
        'ajax' => true,
        'services' => [MOODLE_OFFICIAL_MOBILE_SERVICE],
    ],
    'core_grades_grader_gradingpanel_scale_store' => [
        'classname' => 'core_grades\\grades\\grader\\gradingpanel\\scale\\external\\store',
        'description' => 'Store the data required to display the grader grading panel for scale-based grading',
        'type' => 'write',
        'ajax' => true,
        'services' => [MOODLE_OFFICIAL_MOBILE_SERVICE],
    ],
    'core_grades_create_gradecategory' => array (
        'classname' => 'core_grades_external',
        'methodname' => 'create_gradecategory',
        'description' => '** DEPRECATED ** Please do not call this function any more. Use core_grades_create_gradecategories.
                                     Create a grade category inside a course gradebook.',
        'type' => 'write',
        'capabilities' => 'moodle/grade:manage',
    ),
    'core_grades_create_gradecategories' => array (
        'classname' => 'core_grades\external\create_gradecategories',
        'description' => 'Create grade categories inside a course gradebook.',
        'type' => 'write',
        'capabilities' => 'moodle/grade:manage',
    ),
    'core_grading_get_definitions' => array(
        'classname' => 'core_grading_external',
        'methodname' => 'get_definitions',
        'description' => 'Get grading definitions',
        'type' => 'read',
    ),
    'core_grading_get_gradingform_instances' => array(
        'classname' => 'core_grading_external',
        'methodname' => 'get_gradingform_instances',
        'description' => 'Get grading form instances',
        'type' => 'read',
    ),
    'core_grading_save_definitions' => array(
        'classname' => 'core_grading_external',
        'methodname' => 'save_definitions',
        'description' => 'Save grading definitions',
        'type' => 'write',
    ),
    'core_group_add_group_members' => array(
        'classname' => 'core_group_external',
        'methodname' => 'add_group_members',
        'classpath' => 'group/externallib.php',
        'description' => 'Adds group members.',
        'type' => 'write',
        'capabilities' => 'moodle/course:managegroups',
    ),
    'core_group_assign_grouping' => array(
        'classname' => 'core_group_external',
        'methodname' => 'assign_grouping',
        'classpath' => 'group/externallib.php',
        'description' => 'Assing groups from groupings',
        'type' => 'write',
    ),
    'core_group_create_groupings' => array(
        'classname' => 'core_group_external',
        'methodname' => 'create_groupings',
        'classpath' => 'group/externallib.php',
        'description' => 'Creates new groupings',
        'type' => 'write',
    ),
    'core_group_create_groups' => array(
        'classname' => 'core_group_external',
        'methodname' => 'create_groups',
        'classpath' => 'group/externallib.php',
        'description' => 'Creates new groups.',
        'type' => 'write',
        'capabilities' => 'moodle/course:managegroups'
    ),
    'core_group_delete_group_members' => array(
        'classname' => 'core_group_external',
        'methodname' => 'delete_group_members',
        'classpath' => 'group/externallib.php',
        'description' => 'Deletes group members.',
        'type' => 'write',
        'capabilities' => 'moodle/course:managegroups'
    ),
    'core_group_delete_groupings' => array(
        'classname' => 'core_group_external',
        'methodname' => 'delete_groupings',
        'classpath' => 'group/externallib.php',
        'description' => 'Deletes all specified groupings.',
        'type' => 'write',
    ),
    'core_group_delete_groups' => array(
        'classname' => 'core_group_external',
        'methodname' => 'delete_groups',
        'classpath' => 'group/externallib.php',
        'description' => 'Deletes all specified groups.',
        'type' => 'write',
        'capabilities' => 'moodle/course:managegroups'
    ),
    'core_group_get_activity_allowed_groups' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_activity_allowed_groups',
        'classpath' => 'group/externallib.php',
        'description' => 'Gets a list of groups that the user is allowed to access within the specified activity.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_group_get_activity_groupmode' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_activity_groupmode',
        'classpath' => 'group/externallib.php',
        'description' => 'Returns effective groupmode used in a given activity.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_group_get_course_groupings' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_course_groupings',
        'classpath' => 'group/externallib.php',
        'description' => 'Returns all groupings in specified course.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_group_get_course_groups' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_course_groups',
        'classpath' => 'group/externallib.php',
        'description' => 'Returns all groups in specified course.',
        'type' => 'read',
        'ajax' => true,
        'capabilities' => 'moodle/course:managegroups',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_group_get_course_user_groups' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_course_user_groups',
        'classpath' => 'group/externallib.php',
        'description' => 'Returns all groups in specified course for the specified user.',
        'type' => 'read',
        'capabilities' => 'moodle/course:managegroups',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_group_get_group_members' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_group_members',
        'classpath' => 'group/externallib.php',
        'description' => 'Returns group members.',
        'type' => 'read',
        'capabilities' => 'moodle/course:managegroups'
    ),
    'core_group_get_groupings' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_groupings',
        'classpath' => 'group/externallib.php',
        'description' => 'Returns groupings details.',
        'type' => 'read',
    ),
    'core_group_get_groups' => array(
        'classname' => 'core_group_external',
        'methodname' => 'get_groups',
        'classpath' => 'group/externallib.php',
        'description' => 'Returns group details.',
        'type' => 'read',
        'capabilities' => 'moodle/course:managegroups'
    ),
    'core_group_unassign_grouping' => array(
        'classname' => 'core_group_external',
        'methodname' => 'unassign_grouping',
        'classpath' => 'group/externallib.php',
        'description' => 'Unassing groups from groupings',
        'type' => 'write',
    ),
    'core_group_update_groupings' => array(
        'classname' => 'core_group_external',
        'methodname' => 'update_groupings',
        'classpath' => 'group/externallib.php',
        'description' => 'Updates existing groupings',
        'type' => 'write',
    ),
    'core_group_update_groups' => array(
        'classname' => 'core_group_external',
        'methodname' => 'update_groups',
        'classpath' => 'group/externallib.php',
        'description' => 'Updates existing groups.',
        'type' => 'write',
        'capabilities' => 'moodle/course:managegroups'
    ),
    'core_message_mute_conversations' => array(
        'classname' => 'core_message_external',
        'methodname' => 'mute_conversations',
        'classpath' => 'message/externallib.php',
        'description' => 'Mutes a list of conversations',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_unmute_conversations' => array(
        'classname' => 'core_message_external',
        'methodname' => 'unmute_conversations',
        'classpath' => 'message/externallib.php',
        'description' => 'Unmutes a list of conversations',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_block_user' => array(
        'classname' => 'core_message_external',
        'methodname' => 'block_user',
        'classpath' => 'message/externallib.php',
        'description' => 'Blocks a user',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_get_contact_requests' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_contact_requests',
        'classpath' => 'message/externallib.php',
        'description' => 'Returns contact requests for a user',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_create_contact_request' => array(
        'classname' => 'core_message_external',
        'methodname' => 'create_contact_request',
        'classpath' => 'message/externallib.php',
        'description' => 'Creates a contact request',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_confirm_contact_request' => array(
        'classname' => 'core_message_external',
        'methodname' => 'confirm_contact_request',
        'classpath' => 'message/externallib.php',
        'description' => 'Confirms a contact request',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_decline_contact_request' => array(
        'classname' => 'core_message_external',
        'methodname' => 'decline_contact_request',
        'classpath' => 'message/externallib.php',
        'description' => 'Declines a contact request',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_get_received_contact_requests_count' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_received_contact_requests_count',
        'classpath' => 'message/externallib.php',
        'description' => 'Gets the number of received contact requests',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_delete_contacts' => array(
        'classname' => 'core_message_external',
        'methodname' => 'delete_contacts',
        'classpath' => 'message/externallib.php',
        'description' => 'Remove contacts from the contact list',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_delete_conversations_by_id' => array(
        'classname' => 'core_message_external',
        'methodname' => 'delete_conversations_by_id',
        'classpath' => 'message/externallib.php',
        'description' => 'Deletes a list of conversations.',
        'type' => 'write',
        'capabilities' => 'moodle/site:deleteownmessage',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_delete_message' => array(
        'classname' => 'core_message_external',
        'methodname' => 'delete_message',
        'classpath' => 'message/externallib.php',
        'description' => 'Deletes a message.',
        'type' => 'write',
        'capabilities' => 'moodle/site:deleteownmessage',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_get_blocked_users' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_blocked_users',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve a list of users blocked',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_data_for_messagearea_search_messages' => array(
        'classname' => 'core_message_external',
        'methodname' => 'data_for_messagearea_search_messages',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve the template data for searching for messages',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_message_search_users' => array(
        'classname' => 'core_message_external',
        'methodname' => 'message_search_users',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve the data for searching for people',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_get_user_contacts' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_user_contacts',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve the contact list',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_get_conversations' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_conversations',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve a list of conversations for a user',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true
    ),
    'core_message_get_conversation' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_conversation',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve a conversation for a user',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true
    ),
    'core_message_get_conversation_between_users' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_conversation_between_users',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve a conversation for a user between another user',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true
    ),
    'core_message_get_self_conversation' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_self_conversation',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve a self-conversation for a user',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true
    ),
    'core_message_get_messages' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_messages',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve a list of messages sent and received by a user (conversations, notifications or both)',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true,
    ),
    'core_message_get_conversation_counts' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_conversation_counts',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve a list of conversation counts, indexed by type.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true,
    ),
    'core_message_get_unread_conversation_counts' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_unread_conversation_counts',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve a list of unread conversation counts, indexed by type.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true,
    ),
    'core_message_get_conversation_members' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_conversation_members',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve a list of members in a conversation',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true,
    ),
    'core_message_get_member_info' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_member_info',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve a user message profiles',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true,
    ),
    'core_message_get_unread_conversations_count' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_unread_conversations_count',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve the count of unread conversations for a given user',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'readonlysession' => true, // We don't modify the session.
    ),
    'core_message_mark_all_notifications_as_read' => array(
        'classname' => 'core_message_external',
        'methodname' => 'mark_all_notifications_as_read',
        'classpath' => 'message/externallib.php',
        'description' => 'Mark all notifications as read for a given user',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_mark_all_conversation_messages_as_read' => array(
        'classname' => 'core_message_external',
        'methodname' => 'mark_all_conversation_messages_as_read',
        'classpath' => 'message/externallib.php',
        'description' => 'Mark all conversation messages as read for a given user',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_mark_message_read' => array(
        'classname' => 'core_message_external',
        'methodname' => 'mark_message_read',
        'classpath' => 'message/externallib.php',
        'description' => 'Mark a single message as read, trigger message_viewed event.',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true,
    ),
    'core_message_mark_notification_read' => array(
        'classname' => 'core_message_external',
        'methodname' => 'mark_notification_read',
        'classpath' => 'message/externallib.php',
        'description' => 'Mark a single notification as read, trigger notification_viewed event.',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true,
    ),
    'core_message_message_processor_config_form' => array(
        'classname' => 'core_message_external',
        'methodname' => 'message_processor_config_form',
        'classpath' => 'message/externallib.php',
        'description' => 'Process the message processor config form',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_get_message_processor' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_message_processor',
        'classpath' => 'message/externallib.php',
        'description' => 'Get a message processor',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_message_search_contacts' => array(
        'classname' => 'core_message_external',
        'methodname' => 'search_contacts',
        'classpath' => 'message/externallib.php',
        'description' => 'Search for contacts',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_send_instant_messages' => array(
        'classname' => 'core_message_external',
        'methodname' => 'send_instant_messages',
        'classpath' => 'message/externallib.php',
        'description' => 'Send instant messages',
        'type' => 'write',
        'capabilities' => 'moodle/site:sendmessage',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_send_messages_to_conversation' => array(
        'classname' => 'core_message_external',
        'methodname' => 'send_messages_to_conversation',
        'classpath' => 'message/externallib.php',
        'description' => 'Send messages to an existing conversation between users',
        'type' => 'write',
        'capabilities' => 'moodle/site:sendmessage',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_get_conversation_messages' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_conversation_messages',
        'classpath' => 'message/externallib.php',
        'description' => 'Retrieve the conversation messages and relevant member information',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_unblock_user' => array(
        'classname' => 'core_message_external',
        'methodname' => 'unblock_user',
        'classpath' => 'message/externallib.php',
        'description' => 'Unblocks a user',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_get_user_notification_preferences' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_user_notification_preferences',
        'classpath' => 'message/externallib.php',
        'description' => 'Get the notification preferences for a given user.',
        'type' => 'read',
        'capabilities' => 'moodle/user:editownmessageprofile',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_get_user_message_preferences' => array(
        'classname' => 'core_message_external',
        'methodname' => 'get_user_message_preferences',
        'classpath' => 'message/externallib.php',
        'description' => 'Get the message preferences for a given user.',
        'type' => 'read',
        'capabilities' => 'moodle/user:editownmessageprofile',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
        'ajax' => true
    ),
    'core_message_set_favourite_conversations' => array(
        'classname' => 'core_message_external',
        'methodname' => 'set_favourite_conversations',
        'classpath' => 'message/externallib.php',
        'description' => 'Mark a conversation or group of conversations as favourites/starred conversations.',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_unset_favourite_conversations' => array(
        'classname' => 'core_message_external',
        'methodname' => 'unset_favourite_conversations',
        'classpath' => 'message/externallib.php',
        'description' => 'Unset a conversation or group of conversations as favourites/starred conversations.',
        'type' => 'write',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_message_delete_message_for_all_users' => array(
        'classname' => 'core_message_external',
        'methodname' => 'delete_message_for_all_users',
        'classpath' => 'message/externallib.php',
        'description' => 'Deletes a message for all users.',
        'type' => 'write',
        'capabilities' => 'moodle/site:deleteanymessage',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_notes_create_notes' => array(
        'classname' => 'core_notes_external',
        'methodname' => 'create_notes',
        'classpath' => 'notes/externallib.php',
        'description' => 'Create notes',
        'type' => 'write',
        'ajax' => true,
        'capabilities' => 'moodle/notes:manage',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_notes_delete_notes' => array(
        'classname' => 'core_notes_external',
        'methodname' => 'delete_notes',
        'classpath' => 'notes/externallib.php',
        'description' => 'Delete notes',
        'type' => 'write',
        'capabilities' => 'moodle/notes:manage',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_notes_get_course_notes' => array(
        'classname' => 'core_notes_external',
        'methodname' => 'get_course_notes',
        'classpath' => 'notes/externallib.php',
        'description' => 'Returns all notes in specified course (or site), for the specified user.',
        'type' => 'read',
        'capabilities' => 'moodle/notes:view',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_notes_get_notes' => array(
        'classname' => 'core_notes_external',
        'methodname' => 'get_notes',
        'classpath' => 'notes/externallib.php',
        'description' => 'Get notes',
        'type' => 'read',
        'capabilities' => 'moodle/notes:view'
    ),
    'core_notes_update_notes' => array(
        'classname' => 'core_notes_external',
        'methodname' => 'update_notes',
        'classpath' => 'notes/externallib.php',
        'description' => 'Update notes',
        'type' => 'write',
        'capabilities' => 'moodle/notes:manage'
    ),
    'core_notes_view_notes' => array(
        'classname' => 'core_notes_external',
        'methodname' => 'view_notes',
        'classpath' => 'notes/externallib.php',
        'description' => 'Simulates the web interface view of notes/index.php: trigger events.',
        'type' => 'write',
        'capabilities' => 'moodle/notes:view',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_output_load_template' => array(
        'classname' => 'core\output\external',
        'methodname' => 'load_template',
        'description' => 'Load a template for a renderable',
        'type' => 'read',
        'loginrequired' => false,
        'ajax' => true,
    ),
    'core_output_load_template_with_dependencies' => array(
        'classname' => 'core\output\external',
        'methodname' => 'load_template_with_dependencies',
        'description' => 'Load a template and its dependencies for a renderable',
        'type' => 'read',
        'loginrequired' => false,
        'ajax' => true,
    ),
    'core_output_load_fontawesome_icon_map' => array(
        'classname' => 'core\output\external',
        'methodname' => 'load_fontawesome_icon_map',
        'description' => 'Load the mapping of names to icons',
        'type' => 'read',
        'loginrequired' => false,
        'ajax' => true,
    ),
    'core_output_load_fontawesome_icon_system_map' => array(
        'classname' => 'core\external\output\icon_system\load_fontawesome_map',
        'description' => 'Load the mapping of moodle pix names to fontawesome icon names',
        'type' => 'read',
        'loginrequired' => false,
        'ajax' => true,
    ),
    // Question related functions.
    'core_question_update_flag' => array(
        'classname'     => 'core_question_external',
        'methodname'    => 'update_flag',
        'description'   => 'Update the flag state of a question attempt.',
        'type'          => 'write',
        'capabilities'  => 'moodle/question:flag',
        'services'      => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_question_submit_tags_form' => array(
        'classname'     => 'core_question_external',
        'methodname'    => 'submit_tags_form',
        'description'   => 'Update the question tags.',
        'type'          => 'write',
        'ajax' => true,
    ),
    'core_question_get_random_question_summaries' => array(
        'classname' => 'core_question_external',
        'methodname' => 'get_random_question_summaries',
        'description' => 'Get the random question set for a criteria',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_rating_get_item_ratings' => array(
        'classname' => 'core_rating_external',
        'methodname' => 'get_item_ratings',
        'description' => 'Retrieve all the ratings for an item.',
        'type' => 'read',
        'capabilities' => 'moodle/rating:view',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_rating_add_rating' => array(
        'classname' => 'core_rating_external',
        'methodname' => 'add_rating',
        'description' => 'Rates an item.',
        'type' => 'write',
        'capabilities' => 'moodle/rating:rate',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_role_assign_roles' => array(
        'classname' => 'core_role_external',
        'methodname' => 'assign_roles',
        'classpath' => 'enrol/externallib.php',
        'description' => 'Manual role assignments.',
        'type' => 'write',
        'capabilities' => 'moodle/role:assign'
    ),
    'core_role_unassign_roles' => array(
        'classname' => 'core_role_external',
        'methodname' => 'unassign_roles',
        'classpath' => 'enrol/externallib.php',
        'description' => 'Manual role unassignments.',
        'type' => 'write',
        'capabilities' => 'moodle/role:assign'
    ),
    'core_search_get_relevant_users' => array(
        'classname' => '\core_search\external',
        'methodname' => 'get_relevant_users',
        'description' => 'Gets relevant users for a search request.',
        'type' => 'read',
        'ajax' => true
    ),
    'core_tag_get_tagindex' => array(
        'classname' => 'core_tag_external',
        'methodname' => 'get_tagindex',
        'description' => 'Gets tag index page for one tag and one tag area',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_tag_get_tags' => array(
        'classname' => 'core_tag_external',
        'methodname' => 'get_tags',
        'description' => 'Gets tags by their ids',
        'type' => 'read',
        'ajax' => true,
    ),
    'core_tag_update_tags' => array(
        'classname' => 'core_tag_external',
        'methodname' => 'update_tags',
        'description' => 'Updates tags',
        'type' => 'write',
        'ajax' => true,
    ),
    'core_tag_get_tagindex_per_area' => array(
        'classname' => 'core_tag_external',
        'methodname' => 'get_tagindex_per_area',
        'description' => 'Gets tag index page per different areas.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_tag_get_tag_areas' => array(
        'classname' => 'core_tag_external',
        'methodname' => 'get_tag_areas',
        'description' => 'Retrieves existing tag areas.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_tag_get_tag_collections' => array(
        'classname' => 'core_tag_external',
        'methodname' => 'get_tag_collections',
        'description' => 'Retrieves existing tag collections.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_tag_get_tag_cloud' => array(
        'classname' => 'core_tag_external',
        'methodname' => 'get_tag_cloud',
        'description' => 'Retrieves a tag cloud for the given collection and/or query search.',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_update_inplace_editable' => array(
        'classname' => 'core_external',
        'methodname' => 'update_inplace_editable',
        'classpath' => 'lib/external/externallib.php',
        'description' => 'Generic service to update title',
        'type' => 'write',
        'loginrequired' => true,
        'ajax' => true,
    ),
    'core_user_add_user_device' => array(
        'classname' => 'core_user_external',
        'methodname' => 'add_user_device',
        'classpath' => 'user/externallib.php',
        'description' => 'Store mobile user devices information for PUSH Notifications.',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_add_user_private_files' => array(
        'classname' => 'core_user_external',
        'methodname' => 'add_user_private_files',
        'classpath' => 'user/externallib.php',
        'description' => 'Copy files from a draft area to users private files area.',
        'type' => 'write',
        'capabilities' => 'moodle/user:manageownfiles',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_create_users' => array(
        'classname' => 'core_user_external',
        'methodname' => 'create_users',
        'classpath' => 'user/externallib.php',
        'description' => 'Create users.',
        'type' => 'write',
        'capabilities' => 'moodle/user:create'
    ),
    'core_user_delete_users' => array(
        'classname' => 'core_user_external',
        'methodname' => 'delete_users',
        'classpath' => 'user/externallib.php',
        'description' => 'Delete users.',
        'type' => 'write',
        'capabilities' => 'moodle/user:delete'
    ),
    'core_user_get_course_user_profiles' => array(
        'classname' => 'core_user_external',
        'methodname' => 'get_course_user_profiles',
        'classpath' => 'user/externallib.php',
        'description' => 'Get course user profiles (each of the profils matching a course id and a user id),.',
        'type' => 'read',
        'capabilities' => 'moodle/user:viewdetails, moodle/user:viewhiddendetails, moodle/course:useremail, moodle/user:update, '
            . 'moodle/site:accessallgroups',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_get_users' => array(
        'classname' => 'core_user_external',
        'methodname' => 'get_users',
        'classpath' => 'user/externallib.php',
        'description' => 'search for users matching the parameters',
        'type' => 'read',
        'capabilities' => 'moodle/user:viewdetails, moodle/user:viewhiddendetails, moodle/course:useremail, moodle/user:update'
    ),
    'core_user_get_users_by_field' => array(
        'classname' => 'core_user_external',
        'methodname' => 'get_users_by_field',
        'classpath' => 'user/externallib.php',
        'description' => 'Retrieve users\' information for a specified unique field - If you want to do a user search, use '
            . 'core_user_get_users() or core_user_search_identity().',
        'type' => 'read',
        'capabilities' => 'moodle/user:viewdetails, moodle/user:viewhiddendetails, moodle/course:useremail, moodle/user:update',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_search_identity' => array(
        'classname' => '\core_user\external\search_identity',
        'description' => 'Return list of users identities matching the given criteria in their name or other identity fields.',
        'type' => 'read',
        'capabilities' => 'moodle/user:viewalldetails',
        'ajax' => true,
        'loginrequired' => true,
    ),
    'core_user_remove_user_device' => array(
        'classname' => 'core_user_external',
        'methodname' => 'remove_user_device',
        'classpath' => 'user/externallib.php',
        'description' => 'Remove a user device from the Moodle database.',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_update_users' => array(
        'classname' => 'core_user_external',
        'methodname' => 'update_users',
        'classpath' => 'user/externallib.php',
        'description' => 'Update users.',
        'type' => 'write',
        'capabilities' => 'moodle/user:update',
        'ajax' => true,
    ),
    'core_user_update_user_preferences' => array(
        'classname' => 'core_user_external',
        'methodname' => 'update_user_preferences',
        'classpath' => 'user/externallib.php',
        'description' => 'Update a user\'s preferences',
        'type' => 'write',
        'capabilities' => 'moodle/user:editownmessageprofile, moodle/user:editmessageprofile',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_view_user_list' => array(
        'classname' => 'core_user_external',
        'methodname' => 'view_user_list',
        'classpath' => 'user/externallib.php',
        'description' => 'Simulates the web-interface view of user/index.php (triggering events),.',
        'type' => 'write',
        'capabilities' => 'moodle/course:viewparticipants',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_view_user_profile' => array(
        'classname' => 'core_user_external',
        'methodname' => 'view_user_profile',
        'classpath' => 'user/externallib.php',
        'description' => 'Simulates the web-interface view of user/view.php and user/profile.php (triggering events),.',
        'type' => 'write',
        'capabilities' => 'moodle/user:viewdetails',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_get_user_preferences' => array(
        'classname' => 'core_user_external',
        'methodname' => 'get_user_preferences',
        'classpath' => 'user/externallib.php',
        'description' => 'Return user preferences.',
        'type' => 'read',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_update_picture' => array(
        'classname' => 'core_user_external',
        'methodname' => 'update_picture',
        'classpath' => 'user/externallib.php',
        'description' => 'Update or delete the user picture in the site',
        'type' => 'write',
        'capabilities' => 'moodle/user:editownprofile, moodle/user:editprofile',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_set_user_preferences' => array(
        'classname' => 'core_user_external',
        'methodname' => 'set_user_preferences',
        'classpath' => 'user/externallib.php',
        'description' => 'Set user preferences.',
        'type' => 'write',
        'capabilities' => 'moodle/site:config',
        'ajax' => true,
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_agree_site_policy' => array(
        'classname' => 'core_user_external',
        'methodname' => 'agree_site_policy',
        'classpath' => 'user/externallib.php',
        'description' => 'Agree the site policy for the current user.',
        'type' => 'write',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_user_get_private_files_info' => array(
        'classname' => 'core_user_external',
        'methodname' => 'get_private_files_info',
        'classpath' => 'user/externallib.php',
        'description' => 'Returns general information about files in the user private files area.',
        'type' => 'read',
        'capabilities' => 'moodle/user:manageownfiles',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),

    // Competencies functions.
    'core_competency_create_competency_framework' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'create_competency_framework',
        'classpath'    => '',
        'description'  => 'Creates new competency frameworks.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencymanage',
        'ajax'         => true,
    ),
    'core_competency_read_competency_framework' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'read_competency_framework',
        'classpath'    => '',
        'description'  => 'Load a summary of a competency framework.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:competencyview',
        'ajax'         => true,
    ),
    'core_competency_duplicate_competency_framework' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'duplicate_competency_framework',
        'classpath'    => '',
        'description'  => 'Duplicate a competency framework.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencymanage',
        'ajax'         => true,
    ),
    'core_competency_delete_competency_framework' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'delete_competency_framework',
        'classpath'    => '',
        'description'  => 'Delete a competency framework.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencymanage',
        'ajax'         => true,
    ),
    'core_competency_update_competency_framework' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'update_competency_framework',
        'classpath'    => '',
        'description'  => 'Update a competency framework.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencymanage',
        'ajax'         => true,
    ),
    'core_competency_list_competency_frameworks' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'list_competency_frameworks',
        'classpath'    => '',
        'description'  => 'Load a list of a competency frameworks.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:competencyview',
        'ajax'         => true,
    ),
    'core_competency_count_competency_frameworks' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'count_competency_frameworks',
        'classpath'    => '',
        'description'  => 'Count a list of a competency frameworks.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:competencyview',
        'ajax'         => true,
    ),
    'core_competency_competency_framework_viewed' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'competency_framework_viewed',
        'classpath'    => '',
        'description'  => 'Log event competency framework viewed',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:competencyview',
        'ajax'         => true,
    ),
    'core_competency_create_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'create_competency',
        'classpath'    => '',
        'description'  => 'Creates new competencies.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencymanage',
        'ajax'         => true,
    ),
    'core_competency_read_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'read_competency',
        'classpath'    => '',
        'description'  => 'Load a summary of a competency.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:competencyview',
        'ajax'         => true,
    ),
    'core_competency_competency_viewed' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'competency_viewed',
        'classpath'    => '',
        'description'  => 'Log event competency viewed',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:competencyview',
        'ajax'         => true,
        'services'     => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_competency_delete_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'delete_competency',
        'classpath'    => '',
        'description'  => 'Delete a competency.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencymanage',
        'ajax'         => true,
    ),
    'core_competency_update_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'update_competency',
        'classpath'    => '',
        'description'  => 'Update a competency.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencymanage',
        'ajax'         => true,
    ),
    'core_competency_list_competencies' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'list_competencies',
        'classpath'    => '',
        'description'  => 'Load a list of a competencies.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:competencyview',
        'ajax'         => true,
    ),
    'core_competency_list_competencies_in_template' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'list_competencies_in_template',
        'classpath'    => '',
        'description'  => 'Load a list of a competencies for a given template.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:competencyview',
        'ajax'         => true,
    ),
    'core_competency_count_competencies' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'count_competencies',
        'classpath'    => '',
        'description'  => 'Count a list of a competencies.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:competencyview',
        'ajax'         => true,
    ),
    'core_competency_count_competencies_in_template' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'count_competencies_in_template',
        'classpath'    => '',
        'description'  => 'Count a list of a competencies for a given template.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:competencyview',
        'ajax'         => true,
    ),
    'core_competency_search_competencies' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'search_competencies',
        'classpath'    => '',
        'description'  => 'Search a list of a competencies.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:competencyview',
        'ajax'         => true,
    ),
    'core_competency_set_parent_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'set_parent_competency',
        'classpath'    => '',
        'description'  => 'Set a new parent for a competency.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencymanage',
        'ajax'         => true,
    ),
    'core_competency_move_up_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'move_up_competency',
        'classpath'    => '',
        'description'  => 'Re-order a competency.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencymanage',
        'ajax'         => true,
    ),
    'core_competency_move_down_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'move_down_competency',
        'classpath'    => '',
        'description'  => 'Re-order a competency.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencymanage',
        'ajax'         => true,
    ),
    'core_competency_list_course_module_competencies' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'list_course_module_competencies',
        'classpath'    => '',
        'description'  => 'List the competencies in a course module',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:coursecompetencyview',
        'ajax'         => true,
    ),
    'core_competency_count_course_module_competencies' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'count_course_module_competencies',
        'classpath'    => '',
        'description'  => 'Count the competencies in a course module',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:coursecompetencyview',
        'ajax'         => true,
    ),
    'core_competency_list_course_competencies' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'list_course_competencies',
        'classpath'    => '',
        'description'  => 'List the competencies in a course',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:coursecompetencyview',
        'ajax'         => true,
        'services'     => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_competency_count_competencies_in_course' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'count_competencies_in_course',
        'classpath'    => '',
        'description'  => 'List the competencies in a course',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:coursecompetencyview',
        'ajax'         => true,
    ),
    'core_competency_count_courses_using_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'count_courses_using_competency',
        'classpath'    => '',
        'description'  => 'List the courses using a competency',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:coursecompetencyview',
        'ajax'         => true,
    ),
    'core_competency_add_competency_to_course' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'add_competency_to_course',
        'classpath'    => '',
        'description'  => 'Add the competency to a course',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:coursecompetencymanage',
        'ajax'         => true,
    ),
    'core_competency_add_competency_to_template' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'add_competency_to_template',
        'classpath'    => '',
        'description'  => 'Add the competency to a template',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:templatemanage',
        'ajax'         => true,
    ),
    'core_competency_remove_competency_from_course' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'remove_competency_from_course',
        'classpath'    => '',
        'description'  => 'Remove a competency from a course',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:coursecompetencymanage',
        'ajax'         => true,
    ),
    'core_competency_set_course_competency_ruleoutcome' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'set_course_competency_ruleoutcome',
        'classpath'    => '',
        'description'  => 'Modify the ruleoutcome value for course competency',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:coursecompetencymanage',
        'ajax'         => true,
    ),
    'core_competency_remove_competency_from_template' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'remove_competency_from_template',
        'classpath'    => '',
        'description'  => 'Remove a competency from a template',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:templatemanage',
        'ajax'         => true,
    ),
    'core_competency_reorder_course_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'reorder_course_competency',
        'classpath'    => '',
        'description'  => 'Move a course competency to a new relative sort order.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:coursecompetencymanage',
        'ajax'         => true,
    ),
    'core_competency_reorder_template_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'reorder_template_competency',
        'classpath'    => '',
        'description'  => 'Move a template competency to a new relative sort order.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:templatemanage',
        'ajax'         => true,
    ),
    'core_competency_create_template' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'create_template',
        'classpath'    => '',
        'description'  => 'Creates new learning plan templates.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:templatemanage',
        'ajax'         => true,
    ),
    'core_competency_duplicate_template' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'duplicate_template',
        'classpath'    => '',
        'description'  => 'Duplicate learning plan template.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:templatemanage',
        'ajax'         => true,
    ),
    'core_competency_read_template' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'read_template',
        'classpath'    => '',
        'description'  => 'Load a summary of a learning plan template.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:templateview',
        'ajax'         => true,
    ),
    'core_competency_delete_template' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'delete_template',
        'classpath'    => '',
        'description'  => 'Delete a learning plan template.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:templatemanage',
        'ajax'         => true,
    ),
    'core_competency_update_template' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'update_template',
        'classpath'    => '',
        'description'  => 'Update a learning plan template.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:templatemanage',
        'ajax'         => true,
    ),
    'core_competency_list_templates' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'list_templates',
        'classpath'    => '',
        'description'  => 'Load a list of a learning plan templates.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:templateview',
        'ajax'         => true,
    ),
    'core_competency_list_templates_using_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'list_templates_using_competency',
        'classpath'    => '',
        'description'  => 'Load a list of a learning plan templates for a given competency.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:templateview',
        'ajax'         => true,
    ),
    'core_competency_count_templates' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'count_templates',
        'classpath'    => '',
        'description'  => 'Count a list of a learning plan templates.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:templateview',
        'ajax'         => true,
    ),
    'core_competency_count_templates_using_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'count_templates_using_competency',
        'classpath'    => '',
        'description'  => 'Count a list of a learning plan templates for a given competency.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:templateview',
        'ajax'         => true,
    ),
    'core_competency_create_plan' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'create_plan',
        'classpath'    => '',
        'description'  => 'Creates a learning plan.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:planmanage',
        'ajax'         => true,
    ),
    'core_competency_update_plan' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'update_plan',
        'classpath'    => '',
        'description'  => 'Updates a learning plan.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:planmanage',
        'ajax'         => true,
    ),
    'core_competency_complete_plan' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'complete_plan',
        'classpath'    => '',
        'description'  => 'Complete learning plan.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:planmanage',
        'ajax'         => true,
    ),
    'core_competency_reopen_plan' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'reopen_plan',
        'classpath'    => '',
        'description'  => 'Reopen learning plan.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:planmanage',
        'ajax'         => true,
    ),
    'core_competency_read_plan' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'read_plan',
        'classpath'    => '',
        'description'  => 'Load a learning plan.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:planviewown',
        'ajax'         => true,
    ),
    'core_competency_delete_plan' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'delete_plan',
        'classpath'    => '',
        'description'  => 'Delete a learning plan.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:planmanage',
        'ajax'         => true,
    ),
    'core_competency_list_user_plans' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'list_user_plans',
        'classpath'    => '',
        'description'  => 'List a user\'s learning plans.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:planviewown',
        'ajax'         => true,
    ),
    'core_competency_list_plan_competencies' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'list_plan_competencies',
        'classpath'    => '',
        'description'  => 'List the competencies in a plan',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:planviewown',
        'ajax'         => true,
    ),
    'core_competency_add_competency_to_plan' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'add_competency_to_plan',
        'classpath'    => '',
        'description'  => 'Add the competency to a learning plan',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:planmanage',
        'ajax'         => true,
    ),
    'core_competency_remove_competency_from_plan' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'remove_competency_from_plan',
        'classpath'    => '',
        'description'  => 'Remove the competency from a learning plan',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:planmanage',
        'ajax'         => true,
    ),
    'core_competency_reorder_plan_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'reorder_plan_competency',
        'classpath'    => '',
        'description'  => 'Move a plan competency to a new relative sort order.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:planmanage',
        'ajax'         => true,
    ),
    'core_competency_plan_request_review' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'plan_request_review',
        'classpath'    => '',
        'description'  => 'Request for a plan to be reviewed.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:planmanagedraft',
        'ajax'         => true,
    ),
    'core_competency_plan_start_review' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'plan_start_review',
        'classpath'    => '',
        'description'  => 'Start the review of a plan.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:planmanage',
        'ajax'         => true,
    ),
    'core_competency_plan_stop_review' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'plan_stop_review',
        'classpath'    => '',
        'description'  => 'Stop the review of a plan.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:planmanage',
        'ajax'         => true,
    ),
    'core_competency_plan_cancel_review_request' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'plan_cancel_review_request',
        'classpath'    => '',
        'description'  => 'Cancel the review of a plan.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:planmanagedraft',
        'ajax'         => true,
    ),
    'core_competency_approve_plan' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'approve_plan',
        'classpath'    => '',
        'description'  => 'Approve a plan.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:planmanage',
        'ajax'         => true,
    ),
    'core_competency_unapprove_plan' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'unapprove_plan',
        'classpath'    => '',
        'description'  => 'Unapprove a plan.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:planmanage',
        'ajax'         => true,
    ),
     'core_competency_template_has_related_data' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'template_has_related_data',
        'classpath'    => '',
        'description'  => 'Check if a template has related data',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:templateview',
        'ajax'         => true,
    ),
    'core_competency_get_scale_values' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'get_scale_values',
        'classpath'    => '',
        'description'  => 'Fetch the values for a specific scale',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:competencymanage',
        'ajax'         => true,
        'services'     => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_competency_add_related_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'add_related_competency',
        'classpath'    => '',
        'description'  => 'Adds a related competency',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencymanage',
        'ajax'         => true,
    ),
    'core_competency_remove_related_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'remove_related_competency',
        'classpath'    => '',
        'description'  => 'Remove a related competency',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencymanage',
        'ajax'         => true,
    ),
    'core_competency_read_user_evidence' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'read_user_evidence',
        'classpath'    => '',
        'description'  => 'Read an evidence of prior learning.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:userevidenceview',
        'ajax'         => true,
    ),
    'core_competency_delete_user_evidence' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'delete_user_evidence',
        'classpath'    => '',
        'description'  => 'Delete an evidence of prior learning.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:userevidencemanageown',
        'ajax'         => true,
    ),
    'core_competency_create_user_evidence_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'create_user_evidence_competency',
        'classpath'    => '',
        'description'  => 'Create an evidence of prior learning relationship with a competency.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:userevidencemanageown, moodle/competency:competencyview',
        'ajax'         => true,
    ),
    'core_competency_delete_user_evidence_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'delete_user_evidence_competency',
        'classpath'    => '',
        'description'  => 'Delete an evidence of prior learning relationship with a competency.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:userevidencemanageown',
        'ajax'         => true,
    ),
    'core_competency_user_competency_cancel_review_request' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'user_competency_cancel_review_request',
        'classpath'    => '',
        'description'  => 'Cancel a review request.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:userevidencemanageown',
        'ajax'         => true,
    ),
    'core_competency_user_competency_request_review' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'user_competency_request_review',
        'classpath'    => '',
        'description'  => 'Request a review.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:userevidencemanageown',
        'ajax'         => true,
    ),
    'core_competency_user_competency_start_review' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'user_competency_start_review',
        'classpath'    => '',
        'description'  => 'Start a review.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencygrade',
        'ajax'         => true,
    ),
    'core_competency_user_competency_stop_review' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'user_competency_stop_review',
        'classpath'    => '',
        'description'  => 'Stop a review.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencygrade',
        'ajax'         => true,
    ),
    'core_competency_user_competency_viewed' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'user_competency_viewed',
        'classpath'    => '',
        'description'  => 'Log the user competency viewed event.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:usercompetencyview',
        'ajax'         => true,
        'services'     => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_competency_user_competency_viewed_in_plan' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'user_competency_viewed_in_plan',
        'classpath'    => '',
        'description'  => 'Log the user competency viewed in plan event.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:usercompetencyview',
        'ajax'         => true,
        'services'     => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_competency_user_competency_viewed_in_course' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'user_competency_viewed_in_course',
        'classpath'    => '',
        'description'  => 'Log the user competency viewed in course event',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:usercompetencyview',
        'ajax'         => true,
        'services'     => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_competency_user_competency_plan_viewed' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'user_competency_plan_viewed',
        'classpath'    => '',
        'description'  => 'Log the user competency plan viewed event.',
        'type'         => 'read',
        'capabilities' => 'moodle/competency:usercompetencyview',
        'ajax'         => true,
        'services'     => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_competency_grade_competency' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'grade_competency',
        'classpath'    => '',
        'description'  => 'Grade a competency.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencygrade',
        'ajax'         => true,
    ),
    'core_competency_grade_competency_in_plan' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'grade_competency_in_plan',
        'classpath'    => '',
        'description'  => 'Grade a competency from the user plan page.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencygrade',
        'ajax'         => true,
    ),
    'core_competency_grade_competency_in_course' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'grade_competency_in_course',
        'classpath'    => '',
        'description'  => 'Grade a competency from the course page.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:competencygrade',
        'ajax'         => true,
        'services'     => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_competency_unlink_plan_from_template' => array(
        'classname'    => 'core_competency\external',
        'methodname'   => 'unlink_plan_from_template',
        'classpath'    => '',
        'description'  => 'Unlink a plan form it template.',
        'type'         => 'write',
        'capabilities' => 'moodle/competency:planmanage',
        'ajax'         => true,
    ),
    'core_competency_template_viewed' => array(
        'classname'     => 'core_competency\external',
        'methodname'    => 'template_viewed',
        'classpath'     => '',
        'description'   => 'Log event template viewed',
        'type'          => 'read',
        'capabilities'  => 'moodle/competency:templateview',
        'ajax'          => true,
    ),
    'core_competency_request_review_of_user_evidence_linked_competencies' => array(
        'classname'     => 'core_competency\external',
        'methodname'    => 'request_review_of_user_evidence_linked_competencies',
        'classpath'     => '',
        'description'   => 'Send user evidence competencies in review',
        'type'          => 'write',
        'capabilities'  => 'moodle/competency:userevidencemanageown',
        'ajax'          => true,
    ),
    'core_competency_update_course_competency_settings' => array(
        'classname'     => 'core_competency\external',
        'methodname'    => 'update_course_competency_settings',
        'classpath'     => '',
        'description'   => 'Update the course competency settings',
        'type'          => 'write',
        'capabilities'  => 'moodle/competency:coursecompetencyconfigure',
        'ajax'          => true,
    ),
    'core_competency_delete_evidence' => array(
        'classname'     => 'core_competency\external',
        'methodname'    => 'delete_evidence',
        'classpath'     => '',
        'description'   => 'Delete an evidence',
        'type'          => 'write',
        'capabilities'  => 'moodle/competency:evidencedelete',
        'ajax'          => true,
        'services'      => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),

    'core_webservice_get_site_info' => array(
        'classname' => 'core_webservice_external',
        'methodname' => 'get_site_info',
        'classpath' => 'webservice/externallib.php',
        'description' => 'Return some site info / user info / list web service functions',
        'type' => 'read',
        'services' => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),

    // Blocks functions.
    'core_block_get_course_blocks' => array(
        'classname'     => 'core_block_external',
        'methodname'    => 'get_course_blocks',
        'description'   => 'Returns blocks information for a course.',
        'type'          => 'read',
        'capabilities'  => '',
        'services'      => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),

    'core_block_get_dashboard_blocks' => array(
        'classname'     => 'core_block_external',
        'methodname'    => 'get_dashboard_blocks',
        'description'   => 'Returns blocks information for the given user dashboard.',
        'type'          => 'read',
        'capabilities'  => '',
        'services'      => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),

    'core_block_fetch_addable_blocks' => array(
        'classname'     => 'core_block\external\fetch_addable_blocks',
        'description'   => 'Returns all addable blocks in a given page.',
        'type'          => 'read',
        'capabilities'  => 'moodle/site:manageblocks',
        'ajax'          => true,
        'services'      => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),

    // Filters functions.
    'core_filters_get_available_in_context' => array(
        'classname'   => 'core_filters\external',
        'methodname'  => 'get_available_in_context',
        'description' => 'Returns the filters available in the given contexts.',
        'type'        => 'read',
        'services'    => array(MOODLE_OFFICIAL_MOBILE_SERVICE),
    ),
    'core_customfield_delete_field' => array(
        'classname'   => 'core_customfield_external',
        'methodname'  => 'delete_field',
        'classpath'   => 'customfield/externallib.php',
        'description' => 'Deletes an entry',
        'type'        => 'write',
        'ajax'        => true,
    ),
    'core_customfield_reload_template' => array(
        'classname'   => 'core_customfield_external',
        'methodname'  => 'reload_template',
        'classpath'   => 'customfield/externallib.php',
        'description' => 'Reloads template',
        'type'        => 'read',
        'ajax'        => true,
    ),
    'core_customfield_create_category' => array(
        'classname'   => 'core_customfield_external',
        'methodname'  => 'create_category',
        'classpath'   => 'customfield/externallib.php',
        'description' => 'Creates a new category',
        'type'        => 'write',
        'ajax'        => true,
    ),
    'core_customfield_delete_category' => array(
        'classname'   => 'core_customfield_external',
        'methodname'  => 'delete_category',
        'classpath'   => 'customfield/externallib.php',
        'description' => 'Deletes a category',
        'type'        => 'write',
        'ajax'        => true,
    ),
    'core_customfield_move_field'   => array(
        'classname'   => 'core_customfield_external',
        'methodname'  => 'move_field',
        'classpath'   => 'customfield/externallib.php',
        'description' => 'Drag and drop',
        'type'        => 'write',
        'ajax'        => true,
    ),
    'core_customfield_move_category' => array(
        'classname'   => 'core_customfield_external',
        'methodname'  => 'move_category',
        'classpath'   => 'customfield/externallib.php',
        'description' => 'Drag and drop categories',
        'type'        => 'write',
        'ajax'        => true,
    ),
    'core_h5p_get_trusted_h5p_file' => [
        'classname'     => 'core_h5p\external',
        'methodname'    => 'get_trusted_h5p_file',
        'classpath'     => '',
        'description'   => 'Get the H5P file cleaned for Mobile App.',
        'type'          => 'read',
        'ajax'          => true,
        'capabilities'  => '',
        'services'      => [MOODLE_OFFICIAL_MOBILE_SERVICE],
    ],
    'core_table_get_dynamic_table_content' => [
        'classname' => 'core_table\external\dynamic\get',
        'description' => 'Get the dynamic table content raw html',
        'type' => 'read',
        'ajax' => true,
        'services' => [MOODLE_OFFICIAL_MOBILE_SERVICE],
    ],
    'core_xapi_statement_post' => [
        'classname'     => 'core_xapi\external\post_statement',
        'classpath'     => '',
        'description'   => 'Post an xAPI statement.',
        'type'          => 'write',
        'ajax'          => true,
        'capabilities'  => '',
        'services'      => [MOODLE_OFFICIAL_MOBILE_SERVICE],
    ],
    'core_contentbank_delete_content' => [
        'classname'     => 'core_contentbank\external\delete_content',
        'classpath'     => '',
        'description'   => 'Delete a content from the content bank.',
        'type'          => 'write',
        'ajax'          => true,
        'capabilities'  => 'moodle/contentbank:deleteanycontent',
    ],
    'core_contentbank_rename_content' => [
        'classname'     => 'core_contentbank\external\rename_content',
        'classpath'     => '',
        'description'   => 'Rename a content in the content bank.',
        'type'          => 'write',
        'ajax'          => true,
        'capabilities'  => 'moodle/contentbank:manageowncontent',
    ],
    'core_contentbank_set_content_visibility' => [
        'classname'     => 'core_contentbank\external\set_content_visibility',
        'classpath'     => '',
        'description'   => 'Set the visibility of a content in the content bank.',
        'type'          => 'write',
        'ajax'          => true,
        'capabilities'  => 'moodle/contentbank:manageowncontent',
    ],
    'core_create_userfeedback_action_record' => [
        'classname'     => 'core\external\record_userfeedback_action',
        'classpath'     => '',
        'description'   => 'Record the action that the user takes in the user feedback notification for future use.',
        'type'          => 'write',
        'ajax'          => true,
        'capabilities'  => '',
    ],
    'core_payment_get_available_gateways' => [
        'classname'   => 'core_payment\external\get_available_gateways',
        'description' => 'Get the list of payment gateways that support the given component/area',
        'type'        => 'read',
        'ajax'        => true,
    ],
);

$services = array(
   'Moodle mobile web service'  => array(
        'functions' => array(), // Unused as we add the service in each function definition, third party services would use this.
        'enabled' => 0,
        'restrictedusers' => 0,
        'shortname' => MOODLE_OFFICIAL_MOBILE_SERVICE,
        'downloadfiles' => 1,
        'uploadfiles' => 1
    ),
);
